% UX_cross_synthesis_CV.m   [DAFXbook, 2nd ed., chapter 8]
% ==== This function performs a cross-synthesis with channel vocoder
%
%--------------------------------------------------------------------------
% This source code is provided without any warranties as published in 
% DAFX book 2nd edition, copyright Wiley & Sons 2011, available at 
% http://www.dafx.de. It may be used for educational purposes and not 
% for commercial applications without further permission.
%--------------------------------------------------------------------------

clear

%----- setting user data -----
[DAFx_in_sou,FS] = wavread('moore_guitar'); % signal for source extraction
DAFx_in_env      = wavread('toms_diner');   % signal for spec. env. extraction
ly       = min(length(DAFx_in_sou), length(DAFx_in_env));  % min signal length
DAFx_out = zeros(ly,1);      % result signal
r        = 0.99;             % sound output normalizing ratio
lp       = [1, -2*r, +r*r];  % low-pass filter used
epsi     = 0.00001;

%----- init bandpass frequencies 
f0 = 10;              % start freq in Hz
f0 = f0/FS *2;        % normalized freq
fac_third = 2^(1/3);  % freq factor for third octave
K  = floor(log(1/f0) / log(fac_third)); % number of bands

%----- performing the vocoding or cross synthesis effect -----
fprintf(1, 'band number (max. %i):\n', K);
tic
for k=1:K
  fprintf(1, '%i ', k);
  f1 = f0 * fac_third;            % upper freq of bandpass
  [b, a] = cheby1(2, 3, [f0 f1]); % Chebyshev-type 1 filter design
  f0 = f1;                        % start freq for next band
  %-- filtering the two signals --
  z_sou    = filter(b, a, DAFx_in_sou);          
  z_env    = filter(b, a, DAFx_in_env);
  rms_env  = sqrt(filter(1, lp, z_env.*z_env)); % RMS value of sound 2
  rms_sou  = sqrt(epsi+filter(1, lp, z_sou.*z_sou));  % with whitening
%  rms_sou = 1.;                             % without whitening
  DAFx_out = DAFx_out + z_sou.*rms_env./rms_sou; % add result to output buffer
end
fprintf(1, '\n');
toc

%----- playing and saving output sound -----
soundsc(DAFx_out, FS)
DAFx_out_norm = r * DAFx_out/max(abs(DAFx_out)); % scale for wav output
wavwrite(DAFx_out_norm, FS, 'CrossCV')
